'use strict';

/* --------------------------------------------------------------
 ckeditor.js 2017-07-19
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2017 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 --------------------------------------------------------------
 */

/**
 * ## CKEditor Widget
 *
 * Use this widget on a parent container to convert all the textareas with the "wysiwyg" class into
 * CKEditor instances at once.
 *
 * Official CKEditor Website: {@link http://ckeditor.com}
 *
 * ### Options
 *
 * **File Browser URL | `data-ckeditor-filebrowser-browse-url` | String | Optional**
 *
 * Provide the default URL of the file browser that is integrated within the CKEditor instance. The default
 * value points is 'includes/ckeditor/filemanager/index.html'.
 *
 * **Base URL | `data-ckeditor-base-href` | String | Optional**
 *
 * The base URL of the CKEditor instance. The default value points to the `http://shop.de/admin` directory.
 *
 * **Enter Mode | `data-ckeditor-enter-mode` | Number | Optional**
 *
 * Define the enter mode of the CKEditor instance. The default value of this option is CKEDITOR.ENTER_BR which
 * means that the editor will use the `<br>` element for every line break. For a list of possible values visit
 * this [CKEditor API reference page](http://docs.cksource.com/ckeditor_api/symbols/CKEDITOR.config.html#.enterMode).
 *
 * **Shift Enter Mode | `data-ckeditor-shift-enter-mode` | Number| Optional**
 *
 * Define the shift-enter mode of the CKEditor instance. The default value of this option is CKEDITOR.ENTER_P which
 * means that the editor will use the `<p>` element for every line break. For a list of possible values visit this
 * [CKEditor API reference page](http://docs.cksource.com/ckeditor_api/symbols/CKEDITOR.config.html#.shiftEnterMode).
 *
 * **Language Code | `data-ckeditor-language` | String | Optional**
 *
 * Provide a language code for the CKEditor instance. The default value comes from the
 * `jse.core.config.get('languageCode')` value which has the active language setting of the current page.
 *
 * ### Example
 *
 * When the page loads the textarea element will be converted into a CKEditor instance.
 *
 * ```html
 * <div data-gx-widget="ckeditor">
 *   <textarea name="my-textarea" class="wysiwyg"></textarea>
 * </div>
 * ```
 *
 * **Important: For CKEditor to work correctly the textarea elements need to have a `name` attribute.**
 *
 * @module Admin/Widgets/ckeditor
 * @requires CKEditor-Library
 */
gx.widgets.module('ckeditor', [], function (data) {

    'use strict';

    // ------------------------------------------------------------------------
    // VARIABLE DEFINITION
    // ------------------------------------------------------------------------

    var
    /**
     * Widget Reference
     *
     * @type {object}
     */
    $this = $(this),


    /**
     * Default Options for Widget
     *
     * @type {object}
     */
    defaults = { // Configuration gets passed to the ckeditor.
        'filebrowserBrowseUrl': 'includes/ckeditor/filemanager/index.html',
        'baseHref': jse.core.config.get('appUrl') + '/admin',
        'enterMode': CKEDITOR.ENTER_BR,
        'shiftEnterMode': CKEDITOR.ENTER_P,
        'language': jse.core.config.get('languageCode'),
        'useRelPath': true
    },


    /**
     * Final Widget Options
     *
     * @type {object}
     */
    options = $.extend(true, {}, defaults, data),


    /**
     * Module Object
     *
     * @type {object}
     */
    module = {},


    /**
     * Editors Selector Object
     *
     * @type {object}
     */
    $editors = null;

    // ------------------------------------------------------------------------
    // INITIALIZATION
    // ------------------------------------------------------------------------

    /**
     * Initialize method of the widget, called by the engine.
     */
    module.init = function (done) {
        if (!options.useRelPath) {
            options.filebrowserBrowseUrl += '?mode=mail';
        }

        $editors = $this.filter('.wysiwyg').add($this.find('.wysiwyg'));

        $editors.each(function () {
            var $self = $(this),
                dataset = $.extend({}, options, $self.data()),
                // Get textarea specific configuration.
            name = $self.attr('name');
            $self.removeClass('wysiwyg');
            CKEDITOR.replace(name, dataset);
        });

        // Event handler for the update event, which is updating the ckeditor with the value
        // of the textarea.
        $this.on('ckeditor.update', function () {
            $editors.each(function () {
                var $self = $(this),
                    name = $self.attr('name'),
                    editor = CKEDITOR ? CKEDITOR.instances[name] : null;

                if (editor) {
                    editor.setData($self.val());
                }
            });
        });

        $this.trigger('widget.initialized', 'ckeditor');

        done();
    };

    // Return data to module engine.
    return module;
});
//# sourceMappingURL=data:application/json;charset=utf8;base64,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
